﻿////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// SimpleNdefParser Class and Test
//
// This classes and Tests are released under by MIT License
//
// Copyright 2019 Office-Fun.com(M.Sonobe)
//
// https://opensource.org/licenses/mit-license.php
// 
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace NdefParser
{
    [TestClass]
    public class UnitTest2_結合テスト
    {
        // URI Record Type Definition At Page9
        readonly byte[] sample1
            = new byte[] { 0xD1, 0x01, 0x08, 0x55, 0x01, 0x6e, 0x66, 0x63, 0x2e, 0x63, 0x6f, 0x6d };

        readonly byte[] sample2
            = new byte[] { 0xD1, 0x01, 0x0D, 0x55, 0x05, 0x2b, 0x33, 0x35, 0x38, 0x39, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37 };

        readonly byte[] sample3
            = new byte[] { 0xD1, 0x01, 0x1F, 0x55, 0x00, 0x6d, 0x6d, 0x73, 0x3a, 0x2f, 0x2f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x64, 0x6f, 0x77, 0x6e, 0x6c, 0x6f, 0x61, 0x64, 0x2e, 0x77, 0x6d, 0x76, };

        /// <summary>
        /// NXP213に書き込んだ実例より　Ndef Messageの例　LockControl
        /// </summary>
        readonly byte[] sample4_MyType2Tag
            = new byte[] { 0x01, 0x03, 0xA0, 0xC0, 0x34, 0x03, 0x1B, 0xD1, 0x01, 0x17, 0x55, 0x00, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x6F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x2D, 0x66, 0x75, 0x6E, 0x2E, 0x63, 0x6F, 0x6D, 0xFE };

        readonly byte[] sample4_MyType2Tag_dash
            = new byte[] { 0xD1, 0x01, 0x17, 0x55, 0x00, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x6F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x2D, 0x66, 0x75, 0x6E, 0x2E, 0x63, 0x6F, 0x6D };


        /// <summary>
        /// Qiita 「NDEFについて理解する」を参照 　Ndef Messageの例　Message TLVの例
        /// </summary>
        readonly byte[] sample5_qiita
            = new byte[] { 0x03, 0x0f, 0xD1, 0x01, 0x0B, 0x55, 0x03, 0x71, 0x69, 0x69, 0x74, 0x61, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0xfe };

        readonly byte[] sample5_qiita_dash
            = new byte[] { 0xD1, 0x01, 0x0B, 0x55, 0x03, 0x71, 0x69, 0x69, 0x74, 0x61, 0x2e, 0x63, 0x6f, 0x6d, 0x2f };

        //// TEXTの例
        //// https://www.slideshare.net/tomorrowkey/ndef-13784268
        //readonly byte[] sample6_slideshare
        //    = new byte[] { 0xD1, 0x01, 0x10, 0x54, 0x02, 0x65, 0x6E, 0x48, 0x65, 0x6C, 0x6C, 0x6F, 0x2C, 0x20, 0x57, 0x6F, 0x72,0x6C,0x64,0x2E };

        // page 6.  Table 4. Example: “Hello, world!” も参考になります。テキストを採用する際は、これも例に入れること。

        [TestMethod]
        public void ParseNdefRecordテスト_Sample1()
        {
            byte firstByte = sample1[0];

            Assert.IsTrue(NdefUtility.GetBit(firstByte, (byte)MessageRecordHearder.BitPos_MessageBegin_MB));
            Assert.IsTrue(NdefUtility.GetBit(firstByte, (byte)MessageRecordHearder.BitPos_MessageEnd_ME));
            Assert.IsFalse(NdefUtility.GetBit(firstByte, (byte)MessageRecordHearder.BitPos_ChunkFlag_CF));
            Assert.IsTrue(NdefUtility.GetBit(firstByte, (byte)MessageRecordHearder.BitPos_ShortRecord_SR));
            Assert.IsFalse(NdefUtility.GetBit(firstByte, (byte)MessageRecordHearder.BitPos_IdLength_IL));

            Assert.AreEqual(1, NdefUtility.GetBitRange_From_2_0(firstByte));


            NdefRecord sample1_record = new NdefRecord(sample1);

            Assert.IsTrue(sample1_record.MessageBegin_MB, "NDEF Record MBの取得に失敗しました");
            Assert.IsTrue(sample1_record.MessageEnd_ME, "NDEF Record MEの取得に失敗しました");
            Assert.IsFalse(sample1_record.ChunkFlag_CF, "NDEF Record CFの取得に失敗しました");
            Assert.IsTrue(sample1_record.ShortRecord_SR, "NDEF Record SRの取得に失敗しました");
            Assert.IsFalse(sample1_record.IdLength_IL, "NDEF Record ILの取得に失敗しました");

            Assert.AreEqual(TNF.NfcForumWellKnownType, sample1_record.TypeNameFormat_TNF);

            Assert.AreEqual(0x01, sample1_record.TypeLength, "Type長さの取得に失敗しました");

            Assert.AreEqual(0x08, sample1_record.PayLoadLength, "ペイロードの長さの取得に失敗しました。");

            Assert.AreEqual(WKT_RTD.URI, sample1_record.RecordTypeDefinition, "ペイロードの長さの取得に失敗しました。");

            Assert.AreEqual(0x01, sample1_record.PayLoad[0], "ペイロード[0]の設定に失敗してます");
            Assert.AreEqual(0x6d, sample1_record.PayLoad[7], "ペイロード[0]の設定に失敗してます");


            string answer = "http://www.nfc.com";

            Assert.AreEqual(answer, sample1_record.URI, "取得したURIデータが異なります。");

            Assert.AreEqual(0, string.Compare(sample1_record.URI, @"http://www.nfc.com"));
        }

        [TestMethod]
        public void ParseNdefRecordテスト_Sample1_Main()
        {
            NdefRecord sample1_record = new NdefRecord(sample1);

            string answer = "http://www.nfc.com";

            Assert.AreEqual(answer, sample1_record.URI, "取得したURIデータが異なります。");

            Assert.AreEqual(0, string.Compare(sample1_record.URI, @"http://www.nfc.com"));
        }

        [TestMethod]
        public void ParseNdefRecordテスト_Sample2()
        {
            NdefRecord sample_record = new NdefRecord(sample2);

            Assert.IsTrue(sample_record.MessageBegin_MB, "NDEF Record MBの取得に失敗しました");
            Assert.IsTrue(sample_record.MessageEnd_ME, "NDEF Record MEの取得に失敗しました");
            Assert.IsFalse(sample_record.ChunkFlag_CF, "NDEF Record CFの取得に失敗しました");
            Assert.IsTrue(sample_record.ShortRecord_SR, "NDEF Record SRの取得に失敗しました");
            Assert.IsFalse(sample_record.IdLength_IL, "NDEF Record ILの取得に失敗しました");

            Assert.AreEqual(TNF.NfcForumWellKnownType, sample_record.TypeNameFormat_TNF);

            Assert.AreEqual(0x01, sample_record.TypeLength, "Type長さの取得に失敗しました");

            Assert.AreEqual(13, sample_record.PayLoadLength, "ペイロードの長さの取得に失敗しました。");

            Assert.AreEqual(WKT_RTD.URI, sample_record.RecordTypeDefinition, "ペイロードの長さの取得に失敗しました。");

            Assert.AreEqual(0x05, sample_record.PayLoad[0], "ペイロード[0]の設定に失敗してます");
            Assert.AreEqual(0x37, sample_record.PayLoad[12], "ペイロード[0]の設定に失敗してます");


            string answer = "tel:+35891234567";

            Assert.AreEqual(answer, sample_record.URI, "取得したURIデータが異なります。");

            Assert.AreEqual(0, string.Compare(sample_record.URI, @"tel:+35891234567"));
        }

        [TestMethod]
        public void ParseNdefRecordテスト_Sample3()
        {
            NdefRecord sample_record = new NdefRecord(sample3);

            Assert.IsTrue(sample_record.MessageBegin_MB, "NDEF Record MBの取得に失敗しました");
            Assert.IsTrue(sample_record.MessageEnd_ME, "NDEF Record MEの取得に失敗しました");
            Assert.IsFalse(sample_record.ChunkFlag_CF, "NDEF Record CFの取得に失敗しました");
            Assert.IsTrue(sample_record.ShortRecord_SR, "NDEF Record SRの取得に失敗しました");
            Assert.IsFalse(sample_record.IdLength_IL, "NDEF Record ILの取得に失敗しました");

            Assert.AreEqual(TNF.NfcForumWellKnownType, sample_record.TypeNameFormat_TNF);

            Assert.AreEqual(0x01, sample_record.TypeLength, "Type長さの取得に失敗しました");

            Assert.AreEqual(31, sample_record.PayLoadLength, "ペイロードの長さの取得に失敗しました。");

            Assert.AreEqual(WKT_RTD.URI, sample_record.RecordTypeDefinition, "ペイロードの長さの取得に失敗しました。");

            Assert.AreEqual(0x00, sample_record.PayLoad[0], "ペイロード[0]の設定に失敗してます");
            Assert.AreEqual(0x76, sample_record.PayLoad[30], "ペイロード[0]の設定に失敗してます");


            string answer = "mms://example.com/download.wmv";

            Assert.AreEqual(answer, sample_record.URI, "取得したURIデータが異なります。");

            Assert.AreEqual(0, string.Compare(sample_record.URI, "mms://example.com/download.wmv"));
        }

        [TestMethod]
        public void MakeNdefRecordテスト_Sample1()
        {
            string orgData = "http://www.nfc.com";


            NdefRecord sample1_ndefData = new NdefRecord(orgData, UriIdentifierCode.Http);

            Assert.AreEqual(sample1[0], sample1_ndefData.RawData[0]);
            Assert.AreEqual(sample1[1], sample1_ndefData.RawData[1]);
            Assert.AreEqual(sample1[2], sample1_ndefData.RawData[2]);
            Assert.AreEqual(sample1[3], sample1_ndefData.RawData[3]);
            Assert.AreEqual(sample1[4], sample1_ndefData.RawData[4]);
            Assert.AreEqual(sample1[5], sample1_ndefData.RawData[5]);
            Assert.AreEqual(sample1[6], sample1_ndefData.RawData[6]);
            Assert.AreEqual(sample1[7], sample1_ndefData.RawData[7]);
            Assert.AreEqual(sample1[8], sample1_ndefData.RawData[8]);
            Assert.AreEqual(sample1[9], sample1_ndefData.RawData[9]);
            Assert.AreEqual(sample1[10], sample1_ndefData.RawData[10]);
            Assert.AreEqual(sample1[11], sample1_ndefData.RawData[11]);
        }

        [TestMethod]
        public void MakeNdefRecordテスト_Sample2()
        {
            string orgData = "+35891234567";

            NdefRecord sample_ndefData = new NdefRecord(orgData, UriIdentifierCode.Tel);

            Assert.AreEqual(sample2[0], sample_ndefData.RawData[0]);
            Assert.AreEqual(sample2[1], sample_ndefData.RawData[1]);
            Assert.AreEqual(sample2[2], sample_ndefData.RawData[2]);
            Assert.AreEqual(sample2[3], sample_ndefData.RawData[3]);


            for (int i = 0; i < sample2.Length; ++i)
            {
                Assert.AreEqual(sample2[i], sample_ndefData.RawData[i], i.ToString() + "番目でエラーが発生しました。");
            }
        }

        [TestMethod]
        public void MakeNdefRecordテスト_Sample3()
        {
            string orgData = "mms://example.com/download.wmv";

            NdefRecord sample_ndefData = new NdefRecord(orgData, UriIdentifierCode.UnAbridgedURI);

            for (int i = 0; i < sample3.Length; ++i)
            {
                Assert.AreEqual(sample3[i], sample_ndefData.RawData[i], i.ToString() + "番目でエラーが発生しました。");
            }
        }

        [TestMethod]
        public void MakeNdefRecordテスト_Sample5dash()
        {
            string orgData = "http://qiita.com/";

            NdefRecord sample_ndefData = new NdefRecord(orgData, UriIdentifierCode.Http);

            for (int i = 0; i < sample5_qiita_dash.Length; ++i)
            {
                Assert.AreEqual(sample5_qiita_dash[i], sample_ndefData.RawData[i], i.ToString() + "番目でエラーが発生しました。");
            }
        }

        [TestMethod]
        public void ParseNdefTLV_Sample4()
        {
            NdefTLV ndefTLV = new NdefTLV(sample4_MyType2Tag);

            Assert.AreEqual("https://office-fun.com", ndefTLV.record.URI, "NdefMeasageの解釈でエラーが発生しました。");
        }

        [TestMethod]
        public void ParseNdefRecord_Sample4()
        {
            NdefRecord sample_ndefData = new NdefRecord(sample4_MyType2Tag_dash);

            Assert.AreEqual("https://office-fun.com", sample_ndefData.URI, "NdefRecodeの解釈でエラーが発生しました。");
        }

        [TestMethod]
        public void ParseNdefTLV_Sample5()
        {
            NdefTLV ndefTLV = new NdefTLV(sample5_qiita);

            Assert.AreEqual("http://qiita.com/", ndefTLV.record.URI, "NdefMeasageの解釈でエラーが発生しました。");
        }

        [TestMethod]
        public void MakeNdefTLVテスト_Sample4()
        {
            string orgData = "https://office-fun.com";

            NdefTLV ndefTLV = new NdefTLV(orgData, UriIdentifierCode.UnAbridgedURI, TLVBlock.LockControlTLV);

            for (int i = 0; i < sample4_MyType2Tag.Length; ++i)
            {
                Assert.AreEqual(sample4_MyType2Tag[i], ndefTLV.RawDataForTLV[i], i.ToString() + "番目でエラーが発生しました。");
            }
        }

        [TestMethod]
        public void MakeNdefTLVテスト_Sample5()
        {
            string orgData = "http://qiita.com/";

            NdefTLV ndefTLV = new NdefTLV(orgData, UriIdentifierCode.Http, TLVBlock.NdefMessage_TLV);

            for (int i = 0; i < sample5_qiita.Length; ++i)
            {
                Assert.AreEqual(sample5_qiita[i], ndefTLV.RawDataForTLV[i], i.ToString() + "番目でエラーが発生しました。");
            }
        }

    }
}
